'use client';

import React, { useState, useEffect } from 'react';
import { useParams } from 'next/navigation';
import { 
  User, 
  Car, 
  Calendar, 
  Clock, 
  MapPin, 
  CreditCard, 
  Users,
  Phone,
  Star
} from 'lucide-react';
import DetailLayout from '@/components/ui/DetailLayout';
import { renderCustomer, renderStatusBadge, transferStatusConfig } from '@/components/ui/TableRenderers';
import type { TransferRequest } from '@/types';
import PermissionGuard from '@/components/guards/PermissionGuard';

interface Comment {
  id: string;
  author: string;
  content: string;
  timestamp: Date;
  type: 'comment' | 'system' | 'status_change';
  isInternal?: boolean;
}

export default function TransferDetailPage() {
  return (
    <PermissionGuard requiredPermissions={['transfer_view']}>
      <TransferDetailPageContent />
    </PermissionGuard>
  );
}

function TransferDetailPageContent() {
  const params = useParams();
  const requestId = params.id as string;
  
  const [request, setRequest] = useState<TransferRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [comments, setComments] = useState<Comment[]>([]);

  useEffect(() => {
    fetchRequestDetails();
    fetchComments();
  }, [requestId]);

  const fetchRequestDetails = async () => {
    try {
      setLoading(true);
      
      // Mock data
      const mockRequest: TransferRequest = {
        id: requestId,
        customerId: 'c1',
        customer: {
          id: 'c1',
          type: 'individual',
          firstName: 'Ahmet',
          lastName: 'Yılmaz',
          phone: '+905551234567',
          email: 'ahmet@email.com',
          isVip: false,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date()
        },
        transferType: 'airport_pickup',
        pickupLocation: 'İstanbul Havalimanı',
        dropoffLocation: 'Sultanahmet Oteli',
        pickupDate: new Date('2025-01-15'),
        pickupTime: '14:30',
        passengers: [
          {
            id: 'tp1',
            firstName: 'Ahmet',
            lastName: 'Yılmaz',
            phone: '+905551234567',
            isChild: false
          },
          {
            id: 'tp2',
            firstName: 'Ayşe',
            lastName: 'Yılmaz',
            phone: '+905551234568',
            isChild: false
          }
        ],
        totalPassengers: 2,
        vehicleType: 'comfort',
        vehicleCount: 1,
        hasLuggage: true,
        needsChildSeat: false,
        isWheelchairAccessible: false,
        status: 'assigned',
        assignedDriver: {
          id: 'd1',
          firstName: 'Mehmet',
          lastName: 'Şoför',
          phone: '+905559876543',
          licenseNumber: 'B123456',
          rating: 4.8,
          isActive: true
        },
        assignedVehicle: {
          id: 'v1',
          brand: 'Mercedes',
          model: 'E-Class',
          year: 2022,
          plateNumber: '34 ABC 123',
          capacity: 4,
          type: 'comfort',
          features: ['Klima', 'GPS', 'Wi-Fi'],
          isActive: true
        },
        estimatedPrice: 250,
        finalPrice: 250,
        currency: 'TRY',
        agentNotes: 'Uçak gecikmesi olabilir, şoföre bildirildi',
        specialRequests: 'Havalimanı çıkışında bekleyecek',
        createdAt: new Date('2025-01-10'),
        updatedAt: new Date('2025-01-12')
      };
      
      setRequest(mockRequest);
    } catch (error) {
      console.error('Error fetching request details:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchComments = async () => {
    const mockComments: Comment[] = [
      {
        id: '1',
        author: 'Sistem',
        content: 'Transfer talebi oluşturuldu',
        timestamp: new Date('2025-01-10T10:00:00'),
        type: 'system'
      }
    ];
    setComments(mockComments);
  };

  const handleAddComment = (content: string, isInternal: boolean) => {
    const newComment: Comment = {
      id: Date.now().toString(),
      author: 'Agent Mehmet',
      content,
      timestamp: new Date(),
      type: 'comment',
      isInternal
    };
    setComments(prev => [...prev, newComment]);
  };

  const handleStatusChange = (newStatus: string) => {
    if (!request) return;
    setRequest(prev => prev ? { ...prev, status: newStatus as any, updatedAt: new Date() } : null);
  };

  const statusOptions = [
    { value: 'pending', label: 'Beklemede' },
    { value: 'quoted', label: 'Fiyat Verildi' },
    { value: 'confirmed', label: 'Onaylandı' },
    { value: 'assigned', label: 'Şoför Atandı' },
    { value: 'in_progress', label: 'Devam Ediyor' },
    { value: 'completed', label: 'Tamamlandı' },
    { value: 'cancelled', label: 'İptal Edildi' }
  ];

  if (loading || !request) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-64"></div>
          <div className="h-64 bg-gray-200 rounded w-96"></div>
        </div>
      </div>
    );
  }

  return (
    <DetailLayout
      title={`Transfer Talebi #${request.id}`}
      subtitle={`${request.pickupLocation} → ${request.dropoffLocation}`}
      backUrl="/transfer/requests"
      recordId={request.id}
      recordType="transfer"
      comments={comments}
      onAddComment={handleAddComment}
      onStatusChange={handleStatusChange}
      statusOptions={statusOptions}
      currentStatus={request.status}
    >
      <div className="space-y-6">
        {/* Status Card */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-4">
            <h3 className="text-lg font-semibold text-gray-900">Transfer Durumu</h3>
            {renderStatusBadge(request.status, transferStatusConfig)}
          </div>
        </div>

        {/* Customer Info */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <User className="h-5 w-5 mr-2" />
            Müşteri Bilgileri
          </h3>
          {renderCustomer(request.customer)}
        </div>

        {/* Transfer Details */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Car className="h-5 w-5 mr-2" />
            Transfer Detayları
          </h3>
          <div className="grid grid-cols-2 gap-6">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-green-600" />
                <span className="text-sm text-gray-500">Alış Noktası</span>
              </div>
              <p className="text-lg font-semibold">{request.pickupLocation}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <MapPin className="h-4 w-4 text-red-600" />
                <span className="text-sm text-gray-500">Bırakış Noktası</span>
              </div>
              <p className="text-lg font-semibold">{request.dropoffLocation}</p>
            </div>
          </div>
          
          <div className="mt-6 grid grid-cols-2 gap-4">
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Calendar className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Tarih</span>
              </div>
              <p className="font-medium">{request.pickupDate.toLocaleDateString('tr-TR')}</p>
            </div>
            <div>
              <div className="flex items-center space-x-2 mb-2">
                <Clock className="h-4 w-4 text-gray-400" />
                <span className="text-sm text-gray-500">Saat</span>
              </div>
              <p className="font-medium">{request.pickupTime}</p>
            </div>
          </div>
        </div>

        {/* Vehicle & Passengers */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Users className="h-5 w-5 mr-2" />
            Araç ve Yolcu Bilgileri
          </h3>
          <div className="grid grid-cols-3 gap-4 mb-4">
            <div>
              <span className="text-gray-500">Yolcu Sayısı:</span>
              <p className="font-medium">{request.totalPassengers} kişi</p>
            </div>
            <div>
              <span className="text-gray-500">Araç Tipi:</span>
              <p className="font-medium">{request.vehicleType}</p>
            </div>
            <div>
              <span className="text-gray-500">Araç Sayısı:</span>
              <p className="font-medium">{request.vehicleCount} araç</p>
            </div>
          </div>
          
          <div className="space-y-2">
            {request.hasLuggage && (
              <div className="flex items-center text-sm text-gray-600">
                <span className="w-2 h-2 bg-blue-500 rounded-full mr-2"></span>
                Bagaj var
              </div>
            )}
            {request.needsChildSeat && (
              <div className="flex items-center text-sm text-gray-600">
                <span className="w-2 h-2 bg-green-500 rounded-full mr-2"></span>
                Çocuk koltuğu gerekli
              </div>
            )}
            {request.isWheelchairAccessible && (
              <div className="flex items-center text-sm text-gray-600">
                <span className="w-2 h-2 bg-purple-500 rounded-full mr-2"></span>
                Tekerlekli sandalye erişimi
              </div>
            )}
          </div>
        </div>

        {/* Passengers List */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Yolcu Listesi</h3>
          <div className="space-y-3">
            {request.passengers.map((passenger, index) => (
              <div key={passenger.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <div>
                  <p className="font-medium text-gray-900">
                    {passenger.firstName} {passenger.lastName}
                    {passenger.isChild && (
                      <span className="ml-2 px-2 py-1 text-xs bg-yellow-100 text-yellow-800 rounded">
                        Çocuk
                      </span>
                    )}
                  </p>
                  {passenger.phone && (
                    <p className="text-sm text-gray-500">{passenger.phone}</p>
                  )}
                </div>
                <span className="text-sm text-gray-400">Yolcu {index + 1}</span>
              </div>
            ))}
          </div>
        </div>

        {/* Driver & Vehicle (if assigned) */}
        {request.assignedDriver && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Atanan Şoför ve Araç</h3>
            <div className="grid grid-cols-2 gap-6">
              <div>
                <h4 className="font-medium text-gray-900 mb-2">Şoför Bilgileri</h4>
                <p className="font-medium">{request.assignedDriver.firstName} {request.assignedDriver.lastName}</p>
                <div className="flex items-center space-x-2 mt-1">
                  <Phone className="h-4 w-4 text-gray-400" />
                  <span className="text-sm text-gray-600">{request.assignedDriver.phone}</span>
                </div>
                {request.assignedDriver.rating && (
                  <div className="flex items-center space-x-1 mt-1">
                    <Star className="h-4 w-4 text-yellow-500 fill-current" />
                    <span className="text-sm text-gray-600">{request.assignedDriver.rating}</span>
                  </div>
                )}
              </div>
              
              {request.assignedVehicle && (
                <div>
                  <h4 className="font-medium text-gray-900 mb-2">Araç Bilgileri</h4>
                  <p className="font-medium">{request.assignedVehicle.brand} {request.assignedVehicle.model}</p>
                  <p className="text-sm text-gray-600">{request.assignedVehicle.year} Model</p>
                  <p className="text-sm font-mono text-blue-600">{request.assignedVehicle.plateNumber}</p>
                  {request.assignedVehicle.features.length > 0 && (
                    <div className="mt-2">
                      <p className="text-xs text-gray-500">Özellikler:</p>
                      <div className="flex flex-wrap gap-1 mt-1">
                        {request.assignedVehicle.features.map((feature, index) => (
                          <span key={index} className="px-2 py-1 text-xs bg-blue-100 text-blue-800 rounded">
                            {feature}
                          </span>
                        ))}
                      </div>
                    </div>
                  )}
                </div>
              )}
            </div>
          </div>
        )}

        {/* Price */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <CreditCard className="h-5 w-5 mr-2" />
            Fiyat Bilgileri
          </h3>
          <div className="space-y-2">
            {request.estimatedPrice && (
              <div className="flex justify-between">
                <span>Tahmini Fiyat:</span>
                <span>{request.estimatedPrice.toLocaleString()} {request.currency}</span>
              </div>
            )}
            {request.finalPrice && (
              <div className="flex justify-between font-bold text-lg border-t pt-2">
                <span>Final Fiyat:</span>
                <span className="text-green-600">{request.finalPrice.toLocaleString()} {request.currency}</span>
              </div>
            )}
          </div>
        </div>

        {/* Special Requests */}
        {request.specialRequests && (
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Özel İstekler</h3>
            <p className="text-gray-700">{request.specialRequests}</p>
          </div>
        )}
      </div>
    </DetailLayout>
  );
}